#define __KERNEL__            /* Dieses Programm laeuft im Kernel Space      */
#define MODULE                /* ... und ist ein Modul!                      */

/* Standard-Includefiles fuer Kernel-Module -------------------------------- */

#include <linux/kernel.h>     /* ... fuer Kernel Code                        */
#include <linux/module.h>     /* ... speziell fuer Module                    */
#include <linux/proc_fs.h>    /* Wir benoetigen das Proc-Dateisystem         */

#include <linux/init.h>       /* ? */
#include <linux/config.h>     /* ? */
#include <linux/errno.h>      /* "bekannte" Fehlermeldungen                  */
#include <linux/major.h>      /* ? */
#include <linux/sched.h>      /* ? */
//#include <linux/malloc.h>
#include <linux/slab.h>       // instead of malloc.h
#include <linux/fcntl.h>      /* ? */
#include <linux/delay.h>      /* ? */
#include <asm/uaccess.h>      // copy_to_user

/* Speziell fuer Zeichentreiber -------------------------------------------- */

#include <linux/fs.h>         /* Definitionen fr Zeichentreiber             */
#include <linux/wrapper.h>    /* Kompatibilitaets-Headerfile                 */



// Eigene Prototypen / Konstanten //

#include "../h/glp_devhw.h"
#include "../h/glp_devcfg.h"
#include "../h/glp_dev_ioctl.h"
#include "../h/glp_private.h"
#include "../h/glp_basicio.h"

#include "../h/glp_rtlstuff.h"
#include "../h/glp_cmdExecute.h"
#include "../h/glp_rtlstuff.h"

/* ------------------------------------------------------------------------- */
/*                                                                           */
/* Globale Variablen des Device Drivers                                      */
/*                                                                           */
/* ------------------------------------------------------------------------- */

/* Ist das Device gerade geoeffnet? Dieser "in use" Zaehler verhindert,      */
/* dass mehrere Anwendungsprogramme gleichzeitig auf den Treiber zugreifen.  */
static int iDevice_Open = 0;
static struct tDevicePrivate aPrivate =
  {
    aIOCfg:
      {
        dLatchDelay: GLP_IOCFG_LATCH_DELAY,
        dBusyDelay:  GLP_IOCFG_BUSY_DELAY,
        dBusyTimeOutCount: GLP_IOCFG_BUSY_TIMEOUT_COUNT,
      },

    aLPTAddr:
      {
        iIOPortAddr:GLP_IOCFG_DEFAULT_BASE,
	iSTPortAddr:GLP_IOCFG_DEFAULT_BASE+1,
	iCTPortAddr:GLP_IOCFG_DEFAULT_BASE+2
      },

    aCmdBuffer:
      {
	aDevState:
	{
		bInterfaceMode: 0,
	},
	dError: 0
       // dReturnValue, aCmd, aDevState will in init_module() //
      },

    dWriteTestCnt: 0x1000
  };

// PROC handles //
// [
struct proc_dir_entry *dev_proc_dir = NULL;             // sub dir
struct proc_dir_entry *dev_proc_version = NULL;         // version of driver
struct proc_dir_entry *dev_proc_config = NULL;          // config of driver
struct proc_dir_entry *dev_proc_debug = NULL;           // debug info
// ]


// module init stuff
// [
void dev_init_private(struct tDevicePrivate* aPrv)
  {
   // initiatize the private section
   // initiaize delay configuration
   // [
   aPrv->aIOCfg.dLatchDelay = GLP_IOCFG_LATCH_DELAY;
   aPrv->aIOCfg.dBusyDelay = GLP_IOCFG_BUSY_DELAY;
   aPrv->aIOCfg.dBusyTimeOutCount = GLP_IOCFG_BUSY_TIMEOUT_COUNT;
   // ]

   // initialize port address
   // [
   aPrv->aLPTAddr.iIOPortAddr = 0x378;
   aPrv->aLPTAddr.iSTPortAddr = 0x379;
   aPrv->aLPTAddr.iCTPortAddr = 0x37a;
   // ]

   // initialize cmd- / and return buffer
   // [
   memset(&aPrv->aCmdBuffer, 0, sizeof(aPrv->aCmdBuffer));
   // ]
  }
// ]


// DEVICE DRIVER BASICS
// [


/******************************************************************************
*                                                                             *
*   device_open:    Diese Funktion wird aufgerufen, wenn ein Anwendungs-      *
*                   Programm den Treiber mit open() oeffnet.                  *
*                                                                             *
*   Parameter:      inode, file: Standard-Parameter fuer Treiber              *
*                                                                             *
*   Rueckgabewert:  0 falls alles geklappt hat :-)                            *
*                                                                             *
******************************************************************************/

static int dev_io_handle_open(struct inode *inode, struct file *file)
  {
    /* Falls benoetigt kann hier die minor device number verwendet werden:     */
    /* unsigned int uiMinor = MINOR(inode->i_rdev);                            */

    printk (DEVICE_NAME ": opening device\n");

    /* Fuer's erste lassen wir den Konkurrierenden Zugriff auf den Treiber     */
    /* nicht zu!                                                               */
    if (iDevice_Open) return -EBUSY;

    /* 'in use' Zaehler erhoehen                                               */
    iDevice_Open++;
    printk(DEVICE_NAME " handle device open (counter: %d)\n", iDevice_Open);

    // make default settings //
//    dev_init_private(&aPrivate);

    // 'usage count' erhoehen: zeigt dem Kernel an, dass dieses Modul benutzt
    MOD_INC_USE_COUNT;

    return 0;
  }


/******************************************************************************
*                                                                             *
*   device_release: Diese Funktion wird aufgerufen, wenn das Anwendungs-      *
*                   Programm das Device schliesst. Sie darf nicht             *
*                   fehlschlagen.                                             *
*                                                                             *
*   Parameter:     inode, file: Konvention fuer Kernel-Module                 *
*                                                                             *
*   Rueckgabewert:  Keiner.                                                   *
*                                                                             *
******************************************************************************/

static int dev_io_handle_release(struct inode *inode, struct file *file)
{
  printk (DEVICE_NAME ": closing device\n");

  /* 'in use' Zaehler wieder erniedrigen                                     */
  if (iDevice_Open <= 0)
    printk(DEVICE_NAME "FIXME: problem with deviceuser management (line %d)\n", __LINE__);
  else
    iDevice_Open --;

  printk(DEVICE_NAME " handle device close(counter: %d)\n", iDevice_Open);

  /* 'usage count' erniedrigen                                               */
  MOD_DEC_USE_COUNT;
  return 0;
}


/******************************************************************************
*                                                                             *
*   device_read:    Diese Funktion wird aufgerufen, wenn das Anwendungs-      *
*                   Programm mit 'read()' Daten aus der Pseudodatei des       *
*                   Treibers liest.                                           *
*                                                                             *
*   Parameter:      indev_io_handle_openode:   Wir lesen von diesem inode...                    *
*                   file:    ... und aus diesem file                          *
*                   buffer:  Puffer, der mit den zu lesenden Daten gefuellt   *
*                            wird.                                            *
*                   length:  Maximale Laenge des Puffers, die nicht ueber-    *
*                            schritten werden darf.                           *
*                                                                             *
*   Rueckgabewert:   Anzahl der Bytes, die im Puffer zur Verfuegung stehen,   *
*                   oder Fehlercode:                                          *
*                   - return == length: Alle Daten wurden uebertragen.        *
*                   - return < length:  Es sind noch Daten uebrig             *
*                   - return == 0:      End of File                           *
*                   - return < 0:       Fehler!                               *
*                                                                             *
******************************************************************************/

static ssize_t dev_io_handle_read(struct file *file, char* buffer, size_t aLen, loff_t* aPos)
{
  int iLen = aLen;
  int iPos = *aPos;
  printk (DEVICE_NAME ": in handle_read(%d bytes to read; pos = %d).\n", iLen, iPos);
  /* unsigned int uiMinor = MINOR(file->f_dentry->d_inode->i_rdev); */

  return 0;
}


/******************************************************************************
*                                                                             *
*   device_write:   Diese Funktion wird aufgerufen, wenn das Anwendungs-      *
*                   Programm Daten auf das Pseudo-File schreibt.              *
*   Parameter:      inode, file, buffer, length                               *
*                                                                             *
*   Rueckgabewert:   ???                                                      *
*                                                                             *
******************************************************************************/

static ssize_t dev_io_handle_write(struct file *file, const char* aBf, size_t aLen, loff_t* aPos)
{
  int i = 0;
  int iLen = aLen;
  int iPos = *aPos;
  printk (DEVICE_NAME ": in handle_write(%d to write, %d = pos)\n", iLen, iPos);
  // do write data //
  for (i=0; i < aLen; i++)
    dev_hw_direct_write(aBf[i]);
  return aLen;
}


/******************************************************************************
*                                                                             *
*   device_ioctl:   Diese Funktion definiert die Funktionen des ioctl()       *
*                   Interfaces.                                               *
*                                                                             *
*   Parameter:      inode, file, buffer, length                               *
*                                                                             *
*   Rueckgabewert:  0 im Erfolgsfall                                          *
*                                                                             *
******************************************************************************/


static int dev_io_handle_ioctl(struct inode *inode, struct file* file, unsigned int aCmd, unsigned long aArg)
{
  int iRes = 0;
  int i = 0;
//  printk (DEVICE_NAME ": in handle_ioctl (Cmd: %d).\n", aCmd);

  switch (aCmd)
    {
     case GLP_IOCTL_SET_LPT_ADR:
        iRes = dev_ioctl_handle_setlptadr((struct tGLPAdrLPT*)aArg);
       break;

     case GLP_IOCTL_GET_LPT_ADR:
        iRes = dev_ioctl_handle_getlptadr((struct tGLPAdrLPT*)aArg);
       break;

     case GLP_IOCTL_EXECUTE_BUFFER:
        iRes = dev_ioctl_handle_bfExecute((struct tGLPCmdBuffer*)aArg);
       break;

     case GLP_IOCTL_GET_DEVCFG:
        iRes = dev_ioctl_handle_get_devcfg((struct tGLPDeviceSettings*)aArg);
       break;

     case GLP_IOCTL_SET_DEVCFG:
        iRes = dev_ioctl_handle_set_devcfg((struct tGLPDeviceSettings*)aArg);
       break;

     case GLP_IOCTL_PARANOYA_TEST:
       printk (DEVICE_NAME "initiate pointer dereferance checks...\n\n");
       for (i=0; i < BIO_CMD_MAXNR; i++)
         {
	   (*aBIOInstructionList[i])();
	 }
       glp_rtl_enable();
       printk (DEVICE_NAME "check passed (I hope)!\n");
       break;

// DEBUG STUFF
/*
// [
     case GLP_IOCTL_SET_SPEED_TEST_LENGTH:
        printk (DEVICE_NAME ": setting write test length to %ld.\n", aArg);
	aPrivate.dWriteTestCnt = aArg;
       break;

     case GLP_IOCTL_TEST_SPEED:
        printk (DEVICE_NAME ": execute write test via ioctl.\n");
        dev_hw_direct_write_test();
       break;
// ]
*/

/*    case SAMPLEDRV_TEST1:
      printk(DEVICE_NAME ": test1 ");
      break;*/

    default:
      printk(DEVICE_NAME ": undefined IOCTL call\n");
    }

  return iRes;
}



// Zuordnung der Funktionsaufrufe
// Nicht implementierte Funktionen werden auf NULL gesetzt.

struct file_operations aFileOperations =
  {
    owner:THIS_MODULE,
    write:dev_io_handle_write,
    read:dev_io_handle_read,
    ioctl:dev_io_handle_ioctl,
    open:dev_io_handle_open,
    release:dev_io_handle_release
  };

// ]

// PROC stuff //
// [

static int dev_proc_version_read(char *page, char **start, off_t off, int count, int *eof, void *data)
  {
    int iLen = 0;
    MOD_INC_USE_COUNT;
    iLen = sprintf(page, "version %s\n", DEVICE_VERSION);
    MOD_DEC_USE_COUNT;
    return iLen;
  }

static int dev_proc_config_read(char *page, char **start, off_t off, int count, int *eof, void *data)
  {
    int iLen = 0;
    MOD_INC_USE_COUNT;
    iLen = sprintf(page, "Baseadr           : 0x%x\nLatchdelay        : %ld\nBusydelay         : %ld\nBusytimeout count : %ld\n", aPrivate.aLPTAddr.iIOPortAddr, aPrivate.aIOCfg.dLatchDelay, aPrivate.aIOCfg.dBusyDelay, aPrivate.aIOCfg.dBusyTimeOutCount);
    MOD_DEC_USE_COUNT;
    return (iLen);
  }

#define DEV_PROC_DBG_ADD_TEXT(Str)\
  {\
    iLen += sprintf(&page[iLen], "%s", Str);\
  }

#define DEV_PROC_DBG_ADD_BOOL(Var, Str)\
  {\
    iLen += sprintf(&page[iLen], "%s : %s\n", Str, (Var)?("true"):("false"));\
  }

#define DEV_PROC_DBG_ADD_STR(Str, Var)\
  {\
    iLen += sprintf(&page[iLen], "%s %s\n", Str, Var);\
  }

#define DEV_PROC_DBG_ADD_NUM(Str, Var)\
  {\
    iLen += sprintf(&page[iLen], "%s %d\n", Str, Var);\
  }


static int dev_proc_debug_read(char* page, char **start, off_t off, int count, int *eof, void *data)
  {
    int iLen = 0;
    MOD_INC_USE_COUNT;
    DEV_PROC_DBG_ADD_TEXT("Die FPGA-Register (Latch 1)\n");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q00, "L1_Q00");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q01, "L1_Q01");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q10, "L1_Q10");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q11, "L1_Q11");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q20, "L1_Q20");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q21, "L1_Q21");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q30, "L1_Q30");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q31, "L1_Q31");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q40, "L1_Q40");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q41, "L1_Q41");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q50, "L1_Q50");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q51, "L1_Q51");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q60, "L1_Q60");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q61, "L1_Q61");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q70, "L1_Q70");
    DEV_PROC_DBG_ADD_BOOL(_L1_Q71, "L1_Q71");

    DEV_PROC_DBG_ADD_TEXT("\nDie FPGA-Register (Latch 2)\n");
// Latch "2" Signale;
    DEV_PROC_DBG_ADD_BOOL(_L2_Q00, "L2_Q00");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q01, "L2_Q01");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q10, "L2_Q10");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q11, "L2_Q11");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q20, "L2_Q20");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q21, "L2_Q21");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q30, "L2_Q30");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q31, "L2_Q31");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q40, "L2_Q40");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q41, "L2_Q41");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q50, "L2_Q50");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q51, "L2_Q51");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q60, "L2_Q60");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q61, "L2_Q61");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q70, "L2_Q70");
    DEV_PROC_DBG_ADD_BOOL(_L2_Q71, "L2_Q71");

    DEV_PROC_DBG_ADD_STR("\nInterface Modus:", (aPrivate.aCmdBuffer.aDevState.bInterfaceMode)?("LATCHMODE"):("CLOCKMODE"));
    DEV_PROC_DBG_ADD_NUM("\ncli() / sti() cnt recursive: ", glp_rtl_disableCounter);

    MOD_DEC_USE_COUNT;
    return (iLen);
  }

 int dev_proc_init()
   {
     // create sub dirrectory //
     dev_proc_dir = proc_mkdir(DEVICE_PROC_SUBDIR_NAME, NULL);
     if (!(dev_proc_dir))
       {
         // unable to create sub directory //
	 return -ENOMEM;
       }

     dev_proc_dir->owner = THIS_MODULE;

     // create "version" entry //
     dev_proc_version = create_proc_read_entry(DEVICE_PROC_FILE_VERSION, 0444, dev_proc_dir, dev_proc_version_read, NULL);
     if (!(dev_proc_version))
       {
         // unable to create version entry //
	 return -ENOMEM;
       }
     dev_proc_version->owner = THIS_MODULE;

#ifdef DEVICE_PROC_FILE_CONFIG
     // create "config" entry //
     dev_proc_config = create_proc_read_entry(DEVICE_PROC_FILE_CONFIG, 0444, dev_proc_dir, dev_proc_config_read, NULL);
     if (!(dev_proc_config))
       {
         // unable to create config entry //
	 return -ENOMEM;
       }
     dev_proc_config->owner = THIS_MODULE;
#endif

#ifdef DEVICE_PROC_FILE_DEBUG
     dev_proc_debug = create_proc_read_entry(DEVICE_PROC_FILE_DEBUG, 0444, dev_proc_dir, dev_proc_debug_read, NULL);
     if (!(dev_proc_debug))
       {
         // unable to create debug entry //
	 return -ENOMEM;
       }
     dev_proc_debug->owner = THIS_MODULE;
#endif

     return (0);
   }


 int dev_proc_done()
   {
     if (dev_proc_version && dev_proc_dir)
       remove_proc_entry(DEVICE_PROC_FILE_VERSION, dev_proc_dir);
     dev_proc_version = 0;

#ifdef DEVICE_PROC_FILE_CONFIG
     if (dev_proc_config && dev_proc_dir)
       remove_proc_entry(DEVICE_PROC_FILE_CONFIG, dev_proc_dir);
     dev_proc_config = 0;
#endif

#ifdef DEVICE_PROC_FILE_DEBUG
     if (dev_proc_debug && dev_proc_dir)
       remove_proc_entry(DEVICE_PROC_FILE_DEBUG, dev_proc_dir);
     dev_proc_debug = 0;
#endif

     if (dev_proc_dir)
       remove_proc_entry(DEVICE_PROC_SUBDIR_NAME, NULL);
     dev_proc_dir = 0;
     return (0);
   }

// ]




// MODULE SECTION
// [

/******************************************************************************
*                                                                             *
*   init_module:    Registrieren des Treibers                                 *
*                                                                             *
*   Parameter:      Keine.                                                    *
*                                                                             *
*   Rueckgabewert:   Fehlerwert:                                              *
*									      *
*                   EIO: Die Major Device Number konnte nicht registriert     *
*                        werden.                                              *
*                   0:   Device wurde erfolgreich registriert.                *
*									      *
******************************************************************************/

int init_module()
{
  int iRes = 0;

  printk (DEVICE_NAME "\n-------------------------------------------------------------------------\n");

  // register the device driver //
  if (register_chrdev(DEVICE_MAJOR,DEVICE_NAME, &aFileOperations))
    {
      printk(DEVICE_NAME ": unable to get major %d\n", DEVICE_MAJOR);
      return -EIO;
    }

  // initialize data structures //
  dev_init_private(&aPrivate);

  // get proc entry //
  dev_proc_init();

  // initialize hardware //
  dev_hw_init();

  // initialize internal structures //
  tBasicIO_construct(&aPrivate.aIO, &aPrivate.aIOCfg);

  // finish //
  printk (DEVICE_NAME ": Device registered with major device number %d.\n", DEVICE_MAJOR);
  printk (DEVICE_NAME "FIXME: initialize all (obsolete too)!\n");


  return iRes;
}


/******************************************************************************
*                                                                             *
*   cleanup_module: Unregistrieren des Devices                                *
*                                                                             *
*   Parameter:      Keine.                                                    *
*                                                                             *
*   Rueckgabewert:   Keiner.                                                  *
*									      *
******************************************************************************/

void cleanup_module()
 {
  int iRes = 0;
  // unregister device //
  printk(DEVICE_NAME "try unregister device driver...\n");
  iRes = unregister_chrdev(DEVICE_MAJOR, DEVICE_NAME);

  /* If there's an error, report it                                          */
  if (iRes < 0)
    printk(DEVICE_NAME ": Error in module_unregister_chrdev: %d\n", iRes);

  // remove proc entry //
  dev_proc_done();

  // finish hardware //
  dev_hw_done();
  printk(DEVICE_NAME ": device driver registered.\n");
 }

// ]

